var InfiniteScroller = InfiniteScroller || {};

InfiniteScroller.Game = function() {};

InfiniteScroller.Game.prototype = {
  
  preload: function() {
      this.game.time.advancedTiming = true;
  },
  
  create: function() {

	this.playerStates = 
	{ 
			Running: 1,
			Jumping: 2,
			Sliding: 3,
			Falling: 4,
			Punching:5,
			JumpingPunching: 6,
			FallingPunching: 7
	}
	
    this.game.world.setBounds(0, 0, 3500, this.game.height);
	this.game.modifier = 1;

    // CONSTANTS
	this.blocksSize = 100;
    //Background values
    this.nearBackgroundSpeed = -100;
    this.farBackgroundSpeed = -50;
    //Player values
    this.playerJumpSpeed = -400;
    this.playerJumpMaxHeight = 120;
    this.punchSpeedMultiplier = 1.5;
    //Ground values
    this.groundHeight = 140;
    this.minOffsetBetweenPlatforms = 175;
    this.maxOffsetBetweenPlatforms = 375;
    this.numberOfPlatforms = 5;
    this.groundSpeed = -200;
    // Y position for the floor creation
    var lowHeight = this.game.height - this.groundHeight / 2;
    var highHeight = this.game.height - this.groundHeight;
    this.minGroundWidth = 150;
    this.maxGroundWidth = 1000;
    this.minGroundWidthObjectAppear = 300;
    //Obstacles
    this.shadowHeight = 65;
    this.numberOfObstacles = 4;
    
    this.groundYpositions = 
    [
    	 lowHeight,
    	 highHeight
    ];
    
    //Scene initialization
    this.cameraPositionVariance = 200;
    //Create background
    this.sky = this.add.tileSprite(this.cameraPositionVariance + 42, 0, this.game.width, this.game.height, 'sky');
    this.farBackground = this.add.tileSprite(this.cameraPositionVariance + 42, 0, this.game.width, this.game.height, 'farBackground');
    this.nearBackground = this.add.tileSprite(this.cameraPositionVariance + 42, 0, this.game.width, this.game.height, 'nearBackground');
    // We have a determined pool of grounds for reusing them once they leave the screen space 
    var ground1 = this.add.tileSprite(300, this.groundYpositions[0], 700, this.groundHeight, 'groundTile');
	var ground2 = this.add.tileSprite(ground1.x + ground1.width + this.minOffsetBetweenPlatforms, this.groundYpositions[0], 100, this.groundHeight, 'groundTile');
	var ground3 = this.add.tileSprite(ground2.x + ground2.width + 180, this.groundYpositions[1], 200, this.groundHeight, 'groundTile');
	var ground4 = this.add.tileSprite(ground3.x + ground3.width + 300, this.groundYpositions[0], 100, this.groundHeight, 'groundTile');
	var ground5 = this.add.tileSprite(ground4.x + ground4.width + 375, this.groundYpositions[1], 300, this.groundHeight, 'groundTile');
    this.grounds = 
    [
    	ground1,
    	ground2,
    	ground3,
    	ground4,
    	ground5
    ];
    //Objects
    this.breakableObstacle = [];
    this.wallObstacle = [];
    this.shadowObstacle = [];

    //create player and walk animation
    this.player = this.game.add.sprite(this.game.width/2, this.game.height-90, 'player');    
    this.player.animationRefresh = 4;
    this.player.animations.add('run', [0,1,2,3,4,5]);
    this.player.animations.add('punch', [6,7,8,9,10,11]);
    this.player.animations.add('fall', [12]);
    this.player.animations.add('jump', [13]);
    this.player.animations.add('fallPunch', [14]);
    this.player.animations.add('jumpPunch', [15]);
    this.player.animations.add('slide', [16]);    
    this.player.state = this.playerStates.Running;
    this.game.world.bringToTop(this.player);
    
    //enable physics on the player and ground
    this.enablePhysics();
    
    //player values
    this.player.body.gravity.y = 1000;
    this.player.anchor.setTo(0.5, 1);
    this.player.body.x = 400;
    this.player.initialJumpHeight = this.player.body.y;
    this.player.secondJumpActive = true;
    this.player.heightActive = true;
    this.player.isAlive = true;

    //play the walking animation
    this.player.animations.play('run', this.player.animationRefresh*this.game.modifier, true);

    //move player with cursor keys
    this.cursors = this.game.input.keyboard.createCursorKeys();

    //sounds
	var musicSelection = Math.trunc(this.randomBetweenValues(5,1));
    this.music = this.game.add.audio('music'+musicSelection);
    this.music.loop = true;
    this.music.volume = 0.6;
    this.music.play();
    
    this.boxBreak = this.game.add.audio('boxBreak');
    this.boxBreak.volume = 2;
    
    this.coin = this.game.add.audio('coin');
	this.coin.volume = 0.3;
	
	this.slideSound = this.game.add.audio('slideSound');
	this.slideSound.volume = 0.3;
	
	this.doubleSound = this.game.add.audio('doubleSound');
	this.doubleSound.volume = 0.3;
	
	this.punchSound = this.game.add.audio('punchSound');
	this.punchSound.loop = true;
	this.punchSound.volume = 0.5;

    this.youDied = this.game.add.audio('youDied');
    this.youDied.volume = 0.7;
    
    //stats
    this.points = 0;
    this.pointsInFloat = 0;
    var style1 = { font: "20px Arial", fill: "#3366cc"};
    this.t1 = this.game.add.text(10, 20, "Points:", style1);
    this.t1.fixedToCamera = true;    

    var style2 = { font: "26px Arial", fill: "#ffffff"};
    this.pointsText = this.game.add.text(80, 16, "0", style2);
    this.refreshPoints();
    this.pointsText.fixedToCamera = true;
    
    //Pause
    this.timePaused = 0;
    this.style3 = { font: "80px Impact", fill: "#ff0000"};
    this.finishText;
    this.finishPointsText;
  },
  

  update: function() 
  {
	this.updateElementsSpeed();
	if (this.game.modifier > 0)
	{
		this.game.modifier += 0.05 * this.game.time.elapsed / 1000;
		this.checkPlayerDeath();
		this.refreshPoints();
		this.moveBackground();
		this.checkGroundsOutOfX();
		this.destroyElementsOutOfX();
		this.updateCollisionsOverlap();
	    this.game.camera.setPosition(this.player.x-this.cameraPositionVariance, this.player.y);
	    this.updateStates();
	    //this.debug();
		
	}
	else
	{
		this.gameOver();
	}
	
    //console.log(this.player.state);
    this.player.body.x = 400;
    this.game.world.bringToTop(this.player);
  },
  
  destroyElementsOutOfX: function()
  {
	  for(var i=0; i <  this.breakableObstacle.length; i++)
	  {
		  if (this.breakableObstacle[i].x < this.game.camera.x - this.breakableObstacle[i].width)
		  {
			  this.breakableObstacle[i].destroy();
			  var item = this.breakableObstacle.splice(i,1);
		  }
	  }
	  
	  for(var i=0; i <  this.wallObstacle.length; i++)
	  {
		  if (this.wallObstacle[i].x < this.game.camera.x - this.wallObstacle[i].width)
		  {
			  this.wallObstacle[i].destroy();
			  var item = this.wallObstacle.splice(i,1);
		  }
	  }
	  
	  for(var i=0; i <  this.shadowObstacle.length; i++)
	  {
		  if (this.shadowObstacle[i].x < this.game.camera.x - this.shadowObstacle[i].width)
		  {
			  this.shadowObstacle[i].destroy();
			  var item = this.shadowObstacle.splice(i,1);
		  }
	  }
  },
  
  updateElementsSpeed: function()
  {
	  this.updateGroupSpeed(this.grounds);
	  this.updateGroupSpeed(this.breakableObstacle);
	  this.updateGroupSpeed(this.wallObstacle);
	  this.updateGroupSpeed(this.shadowObstacle);
  },
  
  updateGroupSpeed: function(group)
  {
	  for(var i=0; i < group.length; i++)
	  {
		  if (this.player.state === this.playerStates.Punching
			  || this.player.state === this.playerStates.FallingPunching
			  || this.player.state === this.playerStates.JumpingPunching)
		  {
			  group[i].body.velocity.x = this.groundSpeed * this.game.modifier * this.punchSpeedMultiplier;
		  }
		  else
		  {
			  group[i].body.velocity.x = this.groundSpeed * this.game.modifier;
		  }
	  }
  },

  updateRunning: function()
  {
	  // Check if the player wants to jump
      if (this.playerJump())
	  {
    	  this.player.state = this.playerStates.Jumping;
		  this.player.animations.play('jump', this.player.animationRefresh*this.game.modifier, false);
	  }
      //Check if the player wants to punch
      else if (this.playerPunch())
	  {
    	  this.player.state = this.playerStates.Punching;
    	  var currentFrame = this.player.animations.currentFrame.index;
    	  this.player.animations.play('punch', this.player.animationRefresh*this.game.modifier, true);
    	  this.player.animations.next(currentFrame+1);
	  }
      // Check if the player wants to slide
      else if (this.playerSlide())
	  {
    	  this.player.state = this.playerStates.Sliding;
    	  this.player.animations.play('slide', this.player.animationRefresh*this.game.modifier, false);
    	  this.player.body.height= 50;
    	  this.player.body.y += 50;
	  }
	  
  },

  updateFalling: function()
  {
	  this.applySecondJump();
	  this.checkGroundCollision();
	  // Check if the player wants to fall and punch
      if (this.playerPunch())
	  {
		  this.player.state = this.playerStates.FallingPunching
		  this.player.animations.play('fallPunch', this.player.animationRefresh*this.game.modifier, false);
	  }
  },
  
  updateJumping: function()
  {
	  this.applyVariableJumpHeight();
	  this.checkGroundCollision();
	  // Check if the player wants to jump and punch
      if (this.playerPunch())
	  {
		  this.player.state = this.playerStates.JumpingPunching;
		  this.player.animations.play('jumpPunch', this.player.animationRefresh*this.game.modifier, false);
	  }
  },
  
  updatePunching: function()
  {
	  // Check if the player wants to jump
      if (this.playerJump())
	  {
		  this.player.state = this.playerStates.JumpingPunching;
		  this.player.animations.play('jumpPunch', this.player.animationRefresh*this.game.modifier, false);
	  }
      // Check if the player stops punching
      else if (!this.playerPunch())
	  {
    	  this.player.state = this.playerStates.Running;
    	  var currentFrame = this.player.animations.currentFrame.index;
    	  this.player.animations.play('run', this.player.animationRefresh*this.game.modifier, true);
    	  this.player.animations.next(currentFrame+1);
	  }
  },
  
  updateSliding: function()
  {
	  // Check if the player wants to jump
      if (this.playerJump())
	  {
		  this.player.state = this.playerStates.Jumping;
		  this.player.animations.play('jump', this.player.animationRefresh*this.game.modifier, false);
		  this.player.body.y -= 49;
    	  this.player.body.height = 98;
	  }
	  // Check if the player stops sliding
      else if (!this.playerSlide())
	  {
    	  this.player.state = this.playerStates.Running;
    	  this.player.animations.play('run', this.player.animationRefresh*this.game.modifier, true);
    	  this.player.body.y -= 49;
    	  this.player.body.height = 98;
	  }
  },
  
  updateJumpingPunching: function()
  {
	  this.applyVariableJumpHeight();
	  this.checkGroundCollision();
	  // Check if the player stops punching
      if (!this.playerPunch())
	  {
		  this.player.state = this.playerStates.Jumping;
		  this.player.animations.play('jump', this.player.animationRefresh*this.game.modifier, false);
	  }
  },
  
  updateFallingPunching: function()
  {
	  this.applySecondJump();
	  this.checkGroundCollision();
	  // Check if the player stops punching
	  if (!this.playerPunch())
	  {
		  this.player.state = this.playerStates.Falling
		  this.player.animations.play('fall', this.player.animationRefresh*this.game.modifier, false);
	  }
  },
  
  updateStates: function()
  {
	  // check if the player falls while running
	  if (this.player.body.velocity.y > 75 
		  && this.player.state != this.playerStates.Falling
		  && this.player.state != this.playerStates.FallingPunching)
	  {
		  if (this.player.state == this.playerStates.Sliding)
		  {
			  this.player.body.y -= 49;
	    	  this.player.body.height = 98;
		  }
		  if (this.cursors.right.isDown)
		  {
			  this.player.state = this.playerStates.FallingPunching;
			  this.player.animations.play('fallPunch', this.player.animationRefresh*this.game.modifier, true);
		  }
		  else
	      {
			  this.player.state = this.playerStates.Falling;
			  this.player.animations.play('fall', this.player.animationRefresh*this.game.modifier, true);
	      } 
	  }
	  switch(this.player.state)
	  {
	  	case this.playerStates.Running:
	  		this.updateRunning();
	  		break;
	  	case this.playerStates.Jumping:
	  		this.updateJumping();
	  		break;
	  	case this.playerStates.Sliding:
	  		this.updateSliding();
	  		break;
	  	case this.playerStates.Falling:
	  		this.updateFalling();
	  		break;
	  	case this.playerStates.Punching:
	  		this.updatePunching();
	  		break;
	  	case this.playerStates.JumpingPunching:
	  		this.updateJumpingPunching();
	  		break;
	  	case this.playerStates.FallingPunching:
	  		this.updateFallingPunching();
	  		break;
	  	default:
	  		console.log("player state not found");
	  		break;
	  }
  },
  
  //show updated stats values
  refreshPoints: function() {
    this.pointsInFloat += 0.001 * this.game.modifier* this.game.time.elapsedMS;
    this.points = Math.trunc(this.pointsInFloat);
	this.pointsText.text = this.points;
	
  },
  
  gameOver: function() {
	  	if (this.player.body.velocity.y > 75)
		{
			 this.player.state = this.playerStates.Falling;
			 this.player.animations.play('fall', this.player.animationRefresh*this.game.modifier, true);
		}
	  	if (this.timePaused === 0)
	  	{
			//Generate message showing points at the end
	  		 this.pointsText.visible = false;
	  		 this.t1.visible = false;
	  		 this.finishPointsText = this.game.add.text(this.game.width/2, this.game.height/2, "Points: "+this.points, this.style3);
	  		 this.finishText = this.game.add.text(this.game.width/2, this.game.height/4, "YOU ARE DEAD", this.style3);
	  		 this.finishText.anchor.set(0.5);
	  		 this.finishPointsText.anchor.set(0.5);
	  		 this.finishText.alpha = 0.1;
	  		 this.finishPointsText.alpha = 0.1;
	  		 this.game.add.tween(this.finishText).to( { alpha: 1 }, 3000, "Cubic", true);
	  		 this.game.add.tween(this.finishPointsText).to( { alpha: 1 }, 3000, "Cubic", true);
			 
			 //Stop looping sound and start end music
			 this.music.stop();
			 this.punchSound.stop();
			 this.youDied.play();
	  	}
	  	else if (this.timePaused >= 5) 
	  	{
	  		this.game.state.restart();
	  	}
	  	this.timePaused += this.game.time.elapsed / 1000;
  },
  
  // Set up the physics for the floor and the player
  enablePhysics: function()
  {
	  this.game.physics.arcade.enable(this.player);
	  
	  for(var i=0; i < this.numberOfPlatforms; i++)
	  {
		  this.game.physics.arcade.enable(this.grounds[i]);
		  this.grounds[i].body.immovable = true;
		  this.grounds[i].body.allowGravity = false;
		  this.grounds[i].body.velocity.x = this.groundSpeed;
	  }
  },
  
  // Check if the platforms go outside of the scene from the left and reset them to the right
  checkGroundsOutOfX: function()
  {
	  for(var i=0; i < this.numberOfPlatforms; i++)
	  {
		  if (this.grounds[i].x < this.game.camera.x - this.grounds[i].width)
		  {
			  var distance = this.randomBetweenValues(this.maxOffsetBetweenPlatforms,this.minOffsetBetweenPlatforms,1);
			  var size = this.randomBetweenValues(this.maxGroundWidth, this.minGroundWidth,0.5);
			  //Assign distance
			  if (i != 0)
			  {
				  this.grounds[i].x = this.grounds[i-1].x + this.grounds[i-1].width + distance;
			  }
			  else
			  {
				  this.grounds[i].x = this.grounds[this.numberOfPlatforms-1].x + this.grounds[this.numberOfPlatforms-1].width + distance;
			  }
			  //Assign height
			  var height = Math.random();
			  if (height > 0.5)
			  {
				  this.grounds[i].y = this.groundYpositions[1];
			  }
			  else
			  {
				  this.grounds[i].y = this.groundYpositions[0];
			  }
			  //Assign size
			  this.grounds[i].body.width = size;
			  this.grounds[i].width = size;
			  //Generate obstacles for the ground
			  var typeOfObstacle = Math.trunc(this.randomBetweenValues(this.numberOfObstacles,0));
			  this.createObstacle(this.grounds[i].x,this.grounds[i].y,this.grounds[i].width,typeOfObstacle);
		  }
	  }
  },
  
  //Create a new obstacle, 0 for breakable, 1 for wall
  createObstacle: function(x,y,width, type)
  {
	  if (width >= this.minGroundWidthObjectAppear)
	  {
		  if (type === 0)
		  {
			  this.breakable(x + width/2 - this.blocksSize/2,y-100);
		  }
		  else if (type === 1)
		  {
			  this.breakable(x + 3*width/4 - this.blocksSize/2,y-200);
			  this.wall(x + 3*width/4 - this.blocksSize/2,y-100);
		  }
		  else if (type === 2)
		  {
			  this.wall(x + width/2 - this.blocksSize/2,y-165);
			  this.shadow(x + width/2 - this.blocksSize/2,y-65);
		  }
		  else if (type === 3)
		  {
			  this.wall(x + 3*width/4 - this.blocksSize/2,y-100);
		  }
	  }
  },
  
  //Random between a max and a min, using the game multiplier
  randomBetweenValues: function(max, min,mul)
  {
	  return (Math.random()*(max - min) + min) * this.game.modifier*mul;
  },
  
//Random between a max and a min
  randomBetweenValues: function(max, min)
  {
	  return (Math.random()*(max - min) + min);
  },
  
  //Moves the background and repositions it when necessary to continue the loop
  moveBackground: function()
  {
	  this.nearBackground.tilePosition.x += this.nearBackgroundSpeed * this.game.time.elapsed/1000;
	  this.farBackground.tilePosition.x += this.farBackgroundSpeed * this.game.time.elapsed/1000;  
  },
  
  // Update collision and overlap of the player with the ground
  updateCollisionsOverlap: function()
  {
	  for(var i=0; i < this.numberOfPlatforms; i++)
	  {
		  this.game.physics.arcade.collide(this.player, this.grounds[i], null, null, this);
		  //this.game.physics.arcade.overlap(this.player, this.grounds[i], this.onFloorOverlap, null, this);
	  }
	  for(var i=0; i < this.breakableObstacle.length; i++)
	  {
		  this.game.physics.arcade.collide(this.player, this.breakableObstacle[i], function(obj1, obj2){this.breakableCollision(obj1, obj2, i);},  null, this);
		  //this.game.physics.arcade.overlap(this.player, this.grounds[i], this.onFloorOverlap, null, this);
	  }
	  for(var i=0; i < this.wallObstacle.length; i++)
	  {
		  this.game.physics.arcade.collide(this.player, this.wallObstacle[i], function(obj1, obj2){this.wallCollision(obj1, obj2, i);},  null, this);
		  //this.game.physics.arcade.overlap(this.player, this.grounds[i], this.onFloorOverlap, null, this);
	  }
  },
  
  // Player jump
  playerJump: function()
  {
	  if(this.cursors.up.isDown && this.player.body.velocity.y === 0 && this.player.body.touching.down)
	  {
		  this.player.body.velocity.y = this.playerJumpSpeed;
		  this.player.initialHeight = this.player.body.y;
		  return true;
	  }
	  return false;
  },
  
  // Player punch
  playerPunch: function()
  {
	  if (this.cursors.right.isDown)
	  {
		  if (this.player.state != this.playerStates.Punching
			  && this.player.state != this.playerStates.FallingPunching
			  && this.player.state != this.playerStates.JumpingPunching)
		  {
			  this.punchSound.play();
		  }
		  return true;
	  }
	  else
	  {
		  this.punchSound.stop();
	  }
	  return false;
  },
  
  //Player slide
  playerSlide: function()
  {
	  if (this.cursors.down.isDown)
	  {
		  if (this.player.state != this.playerStates.Sliding)
		  {
			  this.slideSound.play();
		  }
		  return true;
	  }
	  else
	  {
		  this.slideSound.stop();
	  }
	  return false;
  },
  
  //Chech if player has collided with ground
  checkGroundCollision: function()
  {
	  if (this.player.body.touching.down)
	  {
		  this.player.state = this.playerStates.Running;
		  this.player.animations.play('run', this.player.animationRefresh*this.game.modifier, true);
		  this.player.secondJumpActive = true;
		  this.player.heightActive = true;
	  }
  },
  
  //Destroy the box if the player is punching it
  breakableCollision: function(obj1, obj2, i)
  {
	  //Destroy when below box and punching
	  if ((this.player.state === this.playerStates.Punching
		  || this.player.state === this.playerStates.JumpingPunching
		  || this.player.state === this.playerStates.FallingPunching) 
		  && this.player.body.y + this.player.body.height > this.breakableObstacle[i].body.y)
	  {
		  this.breakableObstacle[i].destroy();
		  var item = this.breakableObstacle.splice(i,1);
		  this.pointsInFloat += 25 * this.game.modifier;
		  this.boxBreak.play();
		  this.coin.play();
	  }
	  //Do nothing when above box
	  else if (this.player.body.y + this.player.body.height > this.breakableObstacle[i].body.y)
	  {
		  this.player.isAlive = false;
	  }

  },
  
  //Kills the player
  wallCollision: function(obj1, obj2, i)
  {
	//Do nothing when above box
	  if (this.player.body.y + this.player.body.height > this.wallObstacle[i].body.y)
	  {
		  this.player.isAlive = false;
	  }
  },
  
  //Lets the player jump higher holding the up
  applyVariableJumpHeight: function()
  {
	  if (this.cursors.up.isUp)
	  {
		  this.player.heightActive = false;
	  }
	  if (this.player.body.y > this.player.initialHeight - this.playerJumpMaxHeight 
			  && this.cursors.up.isDown
			  && this.player.secondJumpActive
			  && this.player.heightActive)
	  {
		  this.player.body.velocity.y = this.playerJumpSpeed;
	  }
  },
  
  //Let the player make a second jump when falling
  applySecondJump: function()
  {
	  if (this.player.secondJumpActive && this.cursors.up.isDown)
	  {
		  this.player.body.velocity.y = this.playerJumpSpeed;
		  if (this.player.state === this.playerStates.Falling)
		  {
			  this.player.state = this.playerStates.Jumping;
			  this.player.animations.play('jump', this.player.animationRefresh*this.game.modifier, false);
		  }
		  else if (this.player.state === this.playerStates.FallingPunching)
		  {
			  this.player.state = this.playerStates.JumpingPunching;
			  this.player.animations.play('jumpPunch', this.player.animationRefresh*this.game.modifier, false);
		  }
		  this.player.secondJumpActive = false;
		  this.doubleSound.play();
		  //In case both punch and jump are pressed at the same time
		  if (this.cursors.right.isDown)
		  {
			  this.punchSound.play();
		  }
	  }
  },
  
  //Checks if the player has died and pauses the game
  checkPlayerDeath: function()
  {
	  if (this.player.body.y > this.game.height || !this.player.isAlive)
	  {
		  this.game.modifier = 0;
	  }
  },
  
  breakable: function(x,y) 
  {
	  	var object = this.game.add.sprite(x, y, 'breakable'); 
	  	this.game.physics.arcade.enable(object);
	  	object.body.immovable = true;
		object.body.allowGravity = false;
		object.body.velocity.x = this.groundSpeed * this.game.modifier;
		this.breakableObstacle.push(object);
  },
  
  wall: function(x,y) 
  {
	  	var object = this.game.add.sprite(x, y, 'wall'); 
	  	this.game.physics.arcade.enable(object);
	  	object.body.immovable = true;
		object.body.allowGravity = false;
		object.body.velocity.x = this.groundSpeed * this.game.modifier;
		this.wallObstacle.push(object);
  },
  
  shadow: function(x,y) 
  {
	  	var object = this.game.add.sprite(x, y, 'shadow'); 
	  	this.game.physics.arcade.enable(object);
	  	object.body.immovable = true;
		object.body.allowGravity = false;
		object.body.velocity.x = this.groundSpeed * this.game.modifier;
		this.shadowObstacle.push(object);
  },

  // Debug
  debug: function()
  {
        //this.game.debug.text(this.game.time.fps || '--', 20, 70, "#00ff00", "40px Courier"); 
	  	this.game.debug.bodyInfo(this.player, 16, 16);
	  	for(var i=0; i < this.breakableObstacle.length; i++)
		{
	  		this.game.debug.body(this.breakableObstacle[i]);
		}
		this.game.debug.body(this.player);
   }
};